/*

Copyright 2019 NXP
All rights reserved.
SPDX-License-Identifier: BSD-3-Clause

*/
/*!
 * @addtogroup GENFSK_Localization
 * @addtogroup AngleOfArrival Angle of Arrival localization feature
 * @ingroup GENFSK_Localization
 * Angle of Arrival measurement.
 * @{
 */

/*! @file
 * GENFSK Localization - Angle of Arrival API.
 */

#ifndef LCL_AOA_H_
#define LCL_AOA_H_


/*! *********************************************************************************
*************************************************************************************
* Public macros
*************************************************************************************
********************************************************************************** */
#ifndef LCL_AOA_NUM_MAX_ANTENNAS
#define LCL_AOA_NUM_MAX_ANTENNAS (4U)
#endif

#ifndef LCL_AOA_MAX_POLARIZATIONS
#define LCL_AOA_MAX_POLARIZATIONS (4U)
#endif

#ifndef LCL_AOA_MAX_REPETITION_PATTERN
#define LCL_AOA_MAX_REPETITION_PATTERN (16U)
#endif

#define LCL_MAX_IQ_VALUE (0x7FF0U) // 2047 << 4
#define LCL_MIN_IQ_VALUE (0x8000U) // -2048 << 4

#define LCL_AOA_PREPROC_DC_OFFSET_REMOVAL           (1UL << 0U)
#define LCL_AOA_PREPROC_CFO_REMOVAL                 (1UL << 1U)
#define LCL_AOA_PREPROC_ANTENNA_CAL_COMPENSATION    (1UL << 2U)
#define LCL_AOA_PREPROC_ALL                         (LCL_AOA_PREPROC_DC_OFFSET_REMOVAL | LCL_AOA_PREPROC_CFO_REMOVAL | LCL_AOA_PREPROC_ANTENNA_CAL_COMPENSATION)

#define LCL_AOA_MAX_PATTERN_SIZE (8U)

/*! *********************************************************************************
*************************************************************************************
* Types
*************************************************************************************
********************************************************************************** */

/* Q15 fixed-point format type */
typedef int16_t Q15_t;

/*! @brief Antenna ids. */
typedef enum
{
    gRF1 = 0,
    gRF2,
    gRF3,
    gRF4,
    gRF5,
    gRF6,
    gRF7,
    gRF8,
    gRF9,
    gRF10,
    gRF11,
    gRF12,
    gRF13,
    gRF14,
    gRF15,
    gRF16,
    gRFInvalid,
}lclAntennaId_t;

/*! @brief Antenna polarization to be used. */
typedef enum 
{
    gAntPol_135 = 0,      /*!< Polarization 135 degrees */
    gAntPol_90,           /*!< Polarization 90 degrees (vertical) */
    gAntPol_45,           /*!< Polarization 45 degrees */
    gAntPol_0,            /*!< Polarization 0 degrees (horizontal) */
    gAntPolDynamic = 255/*!< Polarization will be changed during capture:  90 deg for 1st pattern repetition, 45 deg for next repetition and so on */
} lclAntennaPolarization_t;

/*! @brief AoA report type. */
typedef enum 
{
    gAoAReportIqRaw              = 0, /*!< Reports all captured raw IQ samples */
    gAoAReportIqFiltered         = 1, /*!< Reports captured IQ filtered (only sampling periods are kept) samples per antenna */
    gAoAReportAngleAndIqFiltered = 2, /*!< Same as previous plus run AoA algorithm and report computed angle */
    gAoAReportAngle              = 3, /*!< Reports AoA algorithm computed angle */
    gAoAReportAngleAndIqRaw      = 4  /*!< Same as 0 plus run AoA algorithm and report computed angle */
} lclAoaReportType_t;

/*! @brief AoA measurement status type. */
typedef enum 
{
    gAoAMeasSuccess            = 0, /*!< Report succesful */
    gAoAMeasFailSaturation,         /*!< Failure encountered due to I or Q sample(s) hitting saturation */
    gAoAMeasFailOutOfMem,           /*!< Failure encountered due to memory allocation */
    gAoAMeasFailDmaError            /*!< Failure encountered due to DMA error */
} lclAoaMeasStatus_t;

/*! @brief IQ capture results per antenna. */
typedef struct 
{
    lclAntennaId_t      antId; /*!< Antenna id. */
    lclAoaMeasStatus_t  antMeasStatus; /*!< Status for capture at this antenna id. */
    uint16_t            numIq; /*!< Number of IQ samples captured at this antenna id. */
    int16_t             *iqBuff_p; /*!< Pointer on IQ samples buffer captured at this antenna id. *///[gSamplingPeriodDuration * gIqSamplesPerBit * 2];
} lclAoaIqCapturePerAnt_t;

/*! @brief DC offset structure. */
typedef struct 
{
    int16_t      meanI; /*!< I average over reference period */
    int16_t      meanQ; /*!< Q average over reference period */
} lclAoaDcOffset_t;

/*! @brief Algorithm output: computed AoA and associated quality metric */
typedef struct 
{
    int16_t      angleAoA; /*!< Computed AoA in degrees (if requested according to reprtType value). Format is Q9.6 */
    int16_t      aqi; /*!< Angle Quality Indicator metric. Format is Q15 */
} lclAlgoOutput_t;

/*! @brief AGC state */
typedef enum
{
    LCL_AOA_AGC_STATE_UNLOCKED = 0,
    LCL_AOA_AGC_STATE_LOCKING,
    LCL_AOA_AGC_STATE_LOCKED,
    LCL_AOA_AGC_STATE_AUTO,
} lclAoaAgcState_t;

/*! @brief AGC mode */
typedef enum
{
    LCL_AOA_AGC_MODE_AUTO = 0, /*!< AGC is in automatic mpode (HW) */ 
    LCL_AOA_AGC_MODE_MANUAL,   /*!< AGC will be manually locked after evaluation phase */  
    LCL_AOA_AGC_MODE_FORCED    /*!< AGC is locked by SW with value provided by the application */
} lclAoaAgcMode_t;

/*! @brief AGC events */
typedef enum
{
    LCL_AOA_AGC_EVENT_CAPTURE_OK = 0,
    LCL_AOA_AGC_EVENT_SATURATION_DETECTED,
    LCL_AOA_AGC_EVENT_INVALID
} lclAoaAgcEvent_t;

/*! @brief DMA Trigger modes */
typedef enum
{
    LCL_AOA_CAPTURE_TRIGGER_AA_MATCH = 0,
    LCL_AOA_CAPTURE_TRIGGER_PATTERN_MATCH,
    LCL_AOA_CAPTURE_TRIGGER_CTE_PRESENT,
    LCL_AOA_CAPTURE_TRIGGER_INVALID
} lclAoaCaptureTriggerType_t;

/*! @brief IQ capture results indication. */
typedef struct 
{
    uint16_t            measId;        /*!< Measurement id */
    lclAoaReportType_t  reportType;    /*!< Report Type */
    int8_t              rssi;          /*!< RSSI at time of capture. */
    int16_t             cfo_est;        /*!< Internal CFO estimation at time of capture. Format is Q9 */
    uint8_t             agc_index;     /*!< AGC index at time of capture. */
    uint8_t             numAnt;        /*!< Number of antenna used. */
    uint64_t            timestamp;     /*!< Timestamp of capture. */
    lclAlgoOutput_t     algoOut[LCL_AOA_MAX_REPETITION_PATTERN]; /*!< Algorithm output. */
    lclAoaMeasStatus_t  measStatus;    /*!< Golbal status for capture for this measurement id. */
    lclAntennaPolarization_t polarConfig; /*!< Antenna polarization used during capture: fixed or dynamic */
    lclAoaAgcState_t    agcState; /*!< AGC state */
    lclAoaDcOffset_t    dcOffset; /*!< Estimated DC offset */
    uint16_t            iqBuffLength; /*!< IQ buffer length in bytes (num IQ * 4) */
    int16_t             *rawIqBuff_p; /*!< Pointer on RAW IQ samples buffer */
    uint16_t            antPatternLength; /*!< numOfpatternRepetition * numAnt */
    lclAoaIqCapturePerAnt_t  ant[LCL_AOA_MAX_REPETITION_PATTERN*LCL_AOA_NUM_MAX_ANTENNAS]; /*!< keep this array at the end! */

} lclAoaCaptureInd_t;

#define LCL_CAPTURE_IND_MSG_SIZE(numOfpatterns) (offsetof(lclAoaCaptureInd_t, ant) + sizeof(lclAoaIqCapturePerAnt_t)*numOfpatterns)


/*!
 * @brief AoA capture callback function pointer type.
 *
 * @param pBuffer The pointer to the lclAoaCaptureInd_t struct
 */
typedef void ( *lclAoACallBack_t ) (lclAoaCaptureInd_t *captureInd_p);

/*!
 * @brief Antenna switching callback function pointer type.
 *
 * @param Antenna Id
 * @param Antenna polarization
 */
typedef void ( *lclAoAAntSwitchCallBack_t ) (lclAntennaId_t antId, lclAntennaPolarization_t antPol);

/*! @brief Configuration for antenna switching. */
typedef struct 
{
    uint8_t             numAnt; /*!< Number of antenna used. */
    uint16_t            numOfpatternRepetition; /*!< Number of times antenna pattern is repeated in a capture. */
    lclAntennaId_t      mainAntenna; /*!< Main antenna to start reception on, detetc AA and hence lock AGC. */
    uint8_t             dAnt; /*!< distance btw antennas in array in lambda/4 steps */
    uint16_t            guardPeriodDuration; /*!< Guard period duration in us. */
    uint16_t            switchingPeriodDuration; /*!< Switching period duration in us. */
    uint16_t            samplingPeriodDuration; /*!< Sampling period duration in us. */
    lclAntennaId_t      antPattern[LCL_AOA_NUM_MAX_ANTENNAS]; /*!< Antenna switching pattern. */
} lclAntennaConfig_t;

/*! @brief Configuration for AoA measurement. */
typedef struct 
{
    uint8_t             sampleRate; /*!< Sample rate in Msamples/s. 4 for KW37/38. */
    uint8_t             nMpSrc; /*!< Number of sources for MP algo (1 to 3) */
    lclAntennaConfig_t  ant_cnf; /*!< Antenna configuration. */
    lclAoAAntSwitchCallBack_t antSwitchCallback; /*!< Callback called when antenna switch is triggered. */
    lclAoACallBack_t    captureCallback; /*!< Callback called when capture is completed. */
    uint16_t            dmaTriggerDelay; /*!< Delay in us between trigger (AA found) and start of IQ capture. */
    lclAoaAgcMode_t     agcMode;         /*!< AGC mode */
    uint8_t             forcedAgcIndex;  /*!< AGC index if forced mode configured. Range is [0-26] */
    bool_t              disableCFOest;      /*!< Disable CFO estimation */
    lclAoaCaptureTriggerType_t captureTrigger; /*!< Trigger used to start DMA capture */
    uint8_t             byteStreamPatternSize; /*!< Size of byte stream pattern to be matched. Possible values are 4,5,6 or 8 bytes. Only relevant when captureTrigger is set to LCL_AOA_CAPTURE_TRIGGER_PATTERN_MATCH */
    uint8_t             byteStreamPattern[LCL_AOA_MAX_PATTERN_SIZE]; /*!< bit stream pattern to be matched. Only relevant when captureTrigger is set LCL_AOA_CAPTURE_TRIGGER_PATTERN_MATCH */
} lclAoaConfig_t;

typedef struct 
{
    uint16_t             measId;  /*!< Measurement request identifier */
    lclAoaReportType_t   reportType; /*!< Report Type */
    lclAntennaPolarization_t polarConfig; /*!< Antenna polarization to be used during the requested capture: fixed or dynamic */
    
} lclAoaStartCapture_t;
/************************************************************************************
*************************************************************************************
* Public prototypes
*************************************************************************************/

/*!
 * @brief API to configure AoA
 *
 * @param config_p pointer to configuration structure 
 * @return the status of the request, success or invalid parameter
 */
lclStatus_t LclAoAConfigureIqCapture(lclAoaConfig_t *config_p);

/*!
 * @brief API to disable AoA
 *
 * @return the status of the request, success or invalid parameter
 */
lclStatus_t LclAoADisableIqCapture(void);

/*!
 * @brief API to request start of AoA capture
 * @param startCapture_p pointer to lclAoaStartCapture_t structure
 */
void LclAoAStartCapture(lclAoaStartCapture_t *startCapture_p);

/*!
 * @brief API to request stop of AoA capture
 *
 */
void LclAoAStopCapture(void);

/*!
 * @brief API to request pre-processing of IQ samples taken during last AoA capture
 * pre-processing steps are:
 *  - DC offset removal
 *  - Antenna array calibration removal
 *  - CFO compensation
 * @param captureInd_p pointer to lclAoaCaptureInd_t result structure
 *  captureInd_p->ant[] is updated by the call to this API
 * @param flags is a bitmap of the required steps to be performed:
 * see LCL_AOA_PREPROC_DC_OFFSET_REMOVAL, LCL_AOA_PREPROC_CFO_REMOVAL, 
 *     LCL_AOA_PREPROC_ANTENNA_CAL_COMPENSATION, LCL_AOA_PREPROC_ALL  
 * @return the status of the request
 */
lclStatus_t LclAoAIqSamplePreProcessing(uint32_t flags, lclAoaCaptureInd_t *captureInd_p);

/*!
 * @brief API to request post-processing of IQ samples taken during AoA capture
 * @param captureInd_p pointer to lclAoaCaptureInd_t result structure
 *        captureInd_p->algoOut[] is populated by the call to this API
 * @return the status of the request
 */
lclStatus_t LclAoAIqSamplePostProcessing(lclAoaCaptureInd_t *captureInd_p);

/*!
 * @brief API to read phase calibration vector
 * @param cal_phase[] array is provided by the application and will be populated by the call to this API
 * @return none
 */
void LclAoAgetPhaseCalibration(Q15_t cal_phase[LCL_AOA_NUM_MAX_ANTENNAS]);

/*!
 * @brief API to set computed phase calibration vector
 * @param cal_phase[] array is provided by the application and will be stored internally when calling this API
 * @return none
 */
void LclAoAsetPhaseCalibration(Q15_t cal_phase[LCL_AOA_NUM_MAX_ANTENNAS]);

/*!
 * @brief API to request computation phase calibration vector
 * @param captureInd_p: pointer to lclAoaCaptureInd_t result structure
 * @param phaseArray: Phase calibration array to store calibration result
 * @return the status of the request
 */
lclStatus_t LclAoAcomputePhaseCalibration(lclAoaCaptureInd_t *captureInd_p, Q15_t phaseArray[LCL_AOA_NUM_MAX_ANTENNAS]);

/*! @} */

#endif /* LCL_AOA_H_ */
